# ***** BEGIN GPL LICENSE BLOCK *****
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ***** END GPL LICENCE BLOCK *****

bl_info = {
    "name": "Precision Drawing Tools",
    "author": "Alan Odom (Clockmender)",
    "version": (1, 0),
    "blender": (2, 80, 0),
    "location": "VIEW3D (SHIFT+P)",
    "description": "Precision Drawing Tools",
    "warning": "",
    "wiki_url": "",
    "category": "Mesh",
}

import bpy
from bpy.types import Operator
from mathutils import Vector
import bmesh
import numpy as np
from math import sin, cos, acos, pi, sqrt
from mathutils.geometry import intersect_point_line

def move_cursor(self, context, mode_wo, mode_op1, mode_op2, mode_pl, x_loc, y_loc, z_loc,
        dis_v, ang_v, per_v, flip_p, flip_a, trim_b, comm_v):
    obj = bpy.context.view_layer.objects.active
    objs = bpy.context.view_layer.objects.selected
    if obj == None:
        self.report({'WARNING'},
                "Select at least 1 Object")
        return
    obj_loc = obj.matrix_world.decompose()[0]
    if obj.mode == 'EDIT':
        bm = bmesh.from_edit_mesh(obj.data)
        verts = [v for v in bm.verts if v.select]
    if (mode_wo == 'NV' or mode_wo == 'EV' or mode_op2 == 'JV' or mode_op1 == 'INTERSECT') and obj.mode != 'EDIT':
        self.report({'WARNING'},
            "This option Only Works in EDIT Mode")
        return
    if comm_v != '':
        mode_op2 = ''
        # Execute command String
        mode = comm_v[0]
        if mode == 'a':
            mode_op1 = 'ABSOLUTE'
        elif mode == 'd':
            mode_op1 = 'DELTA'
        elif mode == 'v':
            mode_op1 = 'DISTANCE'
        elif mode == 'p':
            mode_op1 = 'PERCENT'
        elif mode == 'n':
            mode_op1 = 'NORMAL'
        elif mode == 'i':
            mode_op1 = 'INTERSECT'
        else:
            mode_op1 = 'BAD'
        vars = comm_v[1:].split(',')
        num = len(vars)
        if mode_op1 == 'BAD':
            self.report({'WARNING'},
                'Command String Invalid: '+mode_op1+' Items: '+str(num))
            return
        elif mode_op1 == 'NORMAL' or mode_op1 == 'INTERSECT':
            vars = ''
        elif mode_op1 == 'PERCENT':
            if num != 1:
                self.report({'WARNING'},
                    'Use Only 1 Variable')
                return
            if '/' in vars[0]:
                try:
                    nvars = vars[0].split('/')
                    per_v = float(nvars[0]) / float(nvars[1])
                except ValueError:
                    per_v = per_v
            else:
                try:
                    per_v = float(vars[0])
                except ValueError:
                    per_v = per_v
        elif mode_op1 == 'DISTANCE':
            if num != 2:
                self.report({'WARNING'},
                    'Use Only 2 Variables')
                return
            try:
                dis_v = float(vars[0])
            except ValueError:
                dis_v = dis_v
            try:
                ang_v = float(vars[1])
            except ValueError:
                ang_v = ang_v
        else:
            if num != 3:
                self.report({'WARNING'},
                    'Use Only 3 Variables')
                return
            try:
                x_loc = float(vars[0])
            except ValueError:
                x_loc = x_loc
            try:
                y_loc = float(vars[1])
            except ValueError:
                y_loc = y_loc
            try:
                z_loc = float(vars[2])
            except ValueError:
                z_loc = z_loc
    #self.stor_v = 'X: '+str(round(x_loc,4))+' Y: '+str(round(y_loc,4))+' Z: '+str(round(z_loc,4))+' D: '+str(round(dis_v,4))+' A: '+str(round(ang_v,4))
    self.comm_v = ''
    if mode_op2 == 'RESET':
        self.x_loc = 0.0
        self.y_loc = 0.0
        self.z_loc = 0.0
        self.dis_v = 0.0
        self.ang_v = 0.0
        self.per_v = 0.0
        #self.stor_v = 'X: '+str(round(x_loc,4))+' Y: '+str(round(y_loc,4))+' Z: '+str(round(z_loc,4))+' D: '+str(round(dis_v,4))+' A: '+str(round(ang_v,4))
        return
    elif mode_op2 == 'JV':
        if len(verts) == 2:
            try:
                nEdge = bm.edges.new([verts[-1],verts[-2]])
                bmesh.update_edit_mesh(obj.data)
                return
            except ValueError:
                self.report({'WARNING'},
                    "Vertices are already connected")
        else:
            self.report({'WARNING'},
                    "Select 2 Vertices")
            return
    elif mode_op2 == 'ANGLE2':
        if obj.mode == 'EDIT':
            if len(bm.select_history) == 2:
                actV,othV = self.checkSelection(2, bm, obj)
            else:
                actV = None
            if actV == None:
                self.report({'WARNING'},
                    "Select 2 Vertices Individually")
                return
        elif obj.mode == 'OBJECT':
            if len(objs) < 2:
                self.report({'WARNING'},
                    "Select 2 Objects")
                return
            objs_s = [ob for ob in objs if ob.name != obj.name]
            actV = obj.matrix_world.decompose()[0]
            othV = objs_s[-1].matrix_world.decompose()[0]
        a1,a2,a3 = self.setMode(self.mode_pl)
        v0 = np.array([actV[a1]+1,actV[a2]]) - np.array([actV[a1],actV[a2]])
        v1 = np.array([othV[a1],othV[a2]]) - np.array([actV[a1],actV[a2]])
        ang = np.rad2deg(np.arctan2(np.linalg.det([v0,v1]),np.dot(v0,v1)))
        if flip_a:
            if ang > 0:
                self.ang_v = ang - 180
            else:
                self.ang_v = ang + 180
        else:
            self.ang_v = ang
        self.dis_v = sqrt((actV[a1]-othV[a1])**2 + (actV[a2]-othV[a2])**2)
        self.x_loc = othV.x-actV.x
        self.y_loc = othV.y-actV.y
        self.z_loc = othV.z-actV.z
        return
    elif mode_op2 == 'ANGLE3':
        if obj.mode == 'EDIT':
            if len(bm.select_history) == 3:
                actV,othV,lstV = self.checkSelection(3, bm, obj)
            else:
                actV = None
            if actV == None:
                self.report({'WARNING'},
                    "Select 3 Vertices Individually")
                return
        elif obj.mode == 'OBJECT':
            objs_s = [ob for ob in objs if ob.name != obj.name]
            if len(objs) < 3:
                self.report({'WARNING'},
                    "Select 3 Objects")
                return
            actV = obj.matrix_world.decompose()[0]
            othV = objs_s[-1].matrix_world.decompose()[0]
            lstV = objs_s[-2].matrix_world.decompose()[0]
        ba = np.array([othV.x,othV.y,othV.z]) - np.array([actV.x,actV.y,actV.z])
        bc = np.array([lstV.x,lstV.y,lstV.z]) - np.array([actV.x,actV.y,actV.z])
        cosA = np.dot(ba, bc) / (np.linalg.norm(ba) * np.linalg.norm(bc))
        ang = np.degrees(np.arccos(cosA))
        if flip_a:
            if ang > 0:
                self.ang_v = ang - 180
            else:
                self.ang_v = ang + 180
        else:
            self.ang_v = ang
        self.dis_v = sqrt((actV.x-othV.x)**2 + (actV.y-othV.y)**2 + (actV.z-othV.z)**2)
        self.x_loc = othV.x-actV.x
        self.y_loc = othV.y-actV.y
        self.z_loc = othV.z-actV.z
        return
    if mode_op1 != 'ABSOLUTE':
        if obj.mode == 'EDIT':
            if len(bm.select_history) >= 1:
                actV = self.checkSelection(1, bm, obj)
            else:
                actV = None
            if actV == None:
                self.report({'WARNING'},
                    "Select at Least 1 Vertex Individually for Delta, Dis/Ang, Percent, Normal & Interset")
                return
            else:
                vector_delta = Vector((actV.x, actV.y, actV.z))
        elif obj.mode == 'OBJECT':
            vector_delta = obj_loc
    if mode_op1 == 'ABSOLUTE':
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                sc.cursor.location = Vector((x_loc,y_loc,z_loc))
        elif mode_wo == 'MV':
            if obj.mode == 'EDIT':
                for v in verts:
                    v.co.x = x_loc - obj_loc.x
                    v.co.y = y_loc - obj_loc.y
                    v.co.z = z_loc - obj_loc.z
                bmesh.ops.remove_doubles(bm, verts=verts, dist=0.001)
                bmesh.update_edit_mesh(obj.data)
            elif obj.mode == 'OBJECT':
                for ob in objs:
                    ob.location = Vector((x_loc,y_loc,z_loc))
        elif mode_wo == 'NV' and obj.mode == 'EDIT':
            vNew = Vector((x_loc,y_loc,z_loc)) - obj_loc
            nVert = bm.verts.new(vNew)
            bmesh.update_edit_mesh(obj.data)
        elif mode_wo == 'EV' and obj.mode == 'EDIT':
            vNew = Vector((x_loc,y_loc,z_loc)) - obj_loc
            nVert = bm.verts.new(vNew)
            for v in verts:
                nEdge = bm.edges.new([v,nVert])
    elif mode_op1 == 'DELTA':
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                if obj.mode == 'EDIT':
                    sc.cursor.location = obj_loc + vector_delta + Vector((x_loc,y_loc,z_loc))
                elif obj.mode == 'OBJECT':
                    sc.cursor.location = obj_loc + Vector((x_loc,y_loc,z_loc))
        elif mode_wo == 'MV':
            if obj.mode == 'EDIT':
                for v in verts:
                    v.co = v.co + Vector((x_loc,y_loc,z_loc))
                bmesh.update_edit_mesh(obj.data)
            elif obj.mode == 'OBJECT':
                for ob in objs:
                    ob.location = ob.location + Vector((x_loc,y_loc,z_loc))
        elif mode_wo == 'NV':
            vNew = vector_delta + Vector((x_loc,y_loc,z_loc))
            nVert = bm.verts.new(vNew)
            bmesh.update_edit_mesh(obj.data)
        elif mode_wo == 'EV':
            for v in verts:
                nVert = bm.verts.new(v.co)
                nVert.co = nVert.co + Vector((x_loc,y_loc,z_loc))
                nEdge = bm.edges.new([v,nVert])
            bmesh.update_edit_mesh(obj.data)
    elif mode_op1 == 'DISTANCE':
        if flip_a:
            if self.ang_v > 0:
                self.ang_v = self.ang_v - 180
            else:
                self.ang_v = self.ang_v + 180
            ang_v = self.ang_v
        vector_delta = Vector((actV.x, actV.y, actV.z))
        a1,a2,a3 = self.setMode(self.mode_pl)
        vector_delta[a1] = vector_delta[a1] + (dis_v * cos(ang_v*pi/180))
        vector_delta[a2] = vector_delta[a2] + (dis_v * sin(ang_v*pi/180))
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                if obj.mode == 'EDIT':
                    sc.cursor.location = obj_loc + vector_delta
                elif obj.mode == 'OBJECT':
                    sc.cursor.location = vector_delta
        elif mode_wo == 'MV':
            if obj.mode == 'EDIT':
                for v in verts:
                    v.co[a1] = v.co[a1] + (dis_v * cos(ang_v*pi/180))
                    v.co[a2] = v.co[a2] + (dis_v * sin(ang_v*pi/180))
                bmesh.update_edit_mesh(obj.data)
            elif obj.mode == 'OBJECT':
                for ob in objs:
                    ob.location[a1] = ob.location[a1] + (dis_v * cos(ang_v*pi/180))
                    ob.location[a2] = ob.location[a2] + (dis_v * sin(ang_v*pi/180))
        elif mode_wo == 'NV':
            nVert = bm.verts.new(vector_delta)
            bmesh.update_edit_mesh(obj.data)
        elif mode_wo == 'EV':
            for v in verts:
                nVert = bm.verts.new(v.co)
                nVert.co[a1] = nVert.co[a1] + (dis_v * cos(ang_v*pi/180))
                nVert.co[a2] = nVert.co[a2] + (dis_v * sin(ang_v*pi/180))
                nEdge = bm.edges.new([v,nVert])
            bmesh.update_edit_mesh(obj.data)
    elif mode_op1 == 'PERCENT':
        if obj.mode == 'EDIT':
            if len(bm.select_history) == 2:
                actV,othV = self.checkSelection(2, bm, obj)
            else:
                actV = None
            if actV == None:
                self.report({'WARNING'},
                    "Select 2 Vertices Individually")
                return
            vector_delta = Vector((actV.x, actV.y, actV.z))
            p1 = np.array([actV.x,actV.y,actV.z])
            p2 = np.array([othV.x,othV.y,othV.z])
        elif obj.mode == 'OBJECT':
            objs = bpy.context.view_layer.objects.selected
            if len(objs) != 2:
                self.report({'WARNING'},
                    "Select Only 2 Objects")
                return
            else:
                p1 = np.array([objs[-1].matrix_world.decompose()[0].x,
                        objs[-1].matrix_world.decompose()[0].y,
                        objs[-1].matrix_world.decompose()[0].z])
                p2 = np.array([objs[-2].matrix_world.decompose()[0].x,
                        objs[-2].matrix_world.decompose()[0].y,
                        objs[-2].matrix_world.decompose()[0].z])
        p4 = np.array([0,0,0])
        p3 = p2-p1
        if flip_p:
            tst = ((p4 + p3) * ((100 - per_v) / 100)) + p1
        else:
            tst = ((p4 + p3) * (per_v / 100)) + p1
        vector_delta = Vector((tst[0],tst[1],tst[2]))
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                if obj.mode == 'EDIT':
                    sc.cursor.location = obj_loc + vector_delta
                elif obj.mode == 'OBJECT':
                    sc.cursor.location = vector_delta
        elif mode_wo == 'MV':
            if obj.mode == 'EDIT':
                bm.select_history[-1].co = vector_delta
                bmesh.update_edit_mesh(obj.data)
            elif obj.mode == 'OBJECT':
                obj.location = vector_delta
        elif mode_wo == 'NV' or mode_wo == 'EV':
            vNew = vector_delta
            nVert = bm.verts.new(vNew)
            if mode_wo == 'EV':
                eVert = bm.select_history[-1]
                nEdge = bm.edges.new([eVert,nVert])
            bmesh.update_edit_mesh(obj.data)
    elif mode_op1 == 'NORMAL':
        if obj.mode == 'EDIT':
            if len(bm.select_history) == 3:
                actV,othV,lstV = self.checkSelection(3, bm, obj)
            else:
                actV = None
            if actV == None:
                self.report({'WARNING'},
                    "Select 3 Vertices Individually")
                return
        elif obj.mode == 'OBJECT':
            objs = bpy.context.view_layer.objects.selected
            if len(objs) != 3:
                self.report({'WARNING'},
                    "Select Only 3 Objects")
                return
            else:
                objs_s = [ob for ob in objs if ob.name != obj.name]
                actV = obj.matrix_world.decompose()[0]
                othV = objs_s[-1].matrix_world.decompose()[0]
                lstV = objs_s[-2].matrix_world.decompose()[0]
        vector_delta = intersect_point_line(actV, othV, lstV)
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                if obj.mode == 'EDIT':
                    sc.cursor.location = obj_loc + vector_delta[0]
                else:
                    sc.cursor.location = vector_delta[0]
        elif mode_wo == 'MV':
            if obj.mode == 'EDIT':
                bm.select_history[-1].co = vector_delta[0]
                bmesh.update_edit_mesh(obj.data)
            elif obj.mode == 'OBJECT':
                obj.location = vector_delta[0]
        elif mode_wo == 'NV' or mode_wo == 'EV':
            vNew = vector_delta[0]
            nVert = bm.verts.new(vNew)
            if mode_wo == 'EV':
                eVert = bm.select_history[-2]
                nEdge = bm.edges.new([eVert,nVert])
            bmesh.update_edit_mesh(obj.data)
    elif mode_op1 == 'INTERSECT':
        if len(bm.select_history) == 4:
            actV,othV,lstV,fstV = self.checkSelection(4, bm, obj)
        else:
            actV = None
        if actV == None:
            self.report({'WARNING'},
                "Select 4 Vertices Individually")
            return
        a1,a2,a3 = self.setMode(self.mode_pl)
        ap1 = (fstV[a1],fstV[a2])
        ap2 = (lstV[a1],lstV[a2])
        bp1 = (othV[a1],othV[a2])
        bp2 = (actV[a1],actV[a2])
        s = np.vstack([ap1,ap2,bp1,bp2])
        h = np.hstack((s, np.ones((4, 1))))
        l1 = np.cross(h[0], h[1])
        l2 = np.cross(h[2], h[3])
        x, y, z = np.cross(l1, l2)
        if z == 0:
            self.report({'WARNING'},
                "Lines Described by these Vertices are Parallel")
            return
        nx = x/z
        nz = y/z
        ly = actV.y
        # Order Vector Delta
        if mode_pl == 'XZ':
            vector_delta = Vector((nx,ly,nz))
        elif mode_pl == 'XY':
            vector_delta = Vector((nx,nz,ly))
        elif mode_pl == 'YZ':
            vector_delta = Vector((ly,nx,nz))
        if mode_wo == 'CU':
            for sc in bpy.data.scenes:
                sc.cursor.location = obj_loc + vector_delta
        elif mode_wo == 'MV':
            if trim_b:
                x1 = bm.select_history[-3].co[0]
                x2 = bm.select_history[-4].co[0]
                x3 = vector_delta[0]
                y1 = bm.select_history[-3].co[1]
                y2 = bm.select_history[-4].co[1]
                y3 = vector_delta[1]
                z1 = bm.select_history[-3].co[2]
                z2 = bm.select_history[-4].co[2]
                z3 = vector_delta[2]
                if sqrt((x1-x3)**2+(y1-y3)**2+(z1-z3)**2) < sqrt((x2-x3)**2+(y2-y3)**2+(z2-z3)**2):
                    bm.select_history[-3].co = vector_delta
                else:
                    bm.select_history[-4].co = vector_delta
                # Second edge
                x1 = bm.select_history[-1].co[0]
                x2 = bm.select_history[-2].co[0]
                y1 = bm.select_history[-1].co[1]
                y2 = bm.select_history[-2].co[1]
                z1 = bm.select_history[-1].co[2]
                z2 = bm.select_history[-2].co[2]
                if sqrt((x1-x3)**2+(y1-y3)**2+(z1-z3)**2) < sqrt((x2-x3)**2+(y2-y3)**2+(z2-z3)**2):
                    bm.select_history[-1].co = vector_delta
                else:
                    bm.select_history[-2].co = vector_delta
            else:
                bm.select_history[-1].co = vector_delta
            bmesh.update_edit_mesh(obj.data)
            bmesh.ops.remove_doubles(bm, verts=verts, dist=0.001)
        elif mode_wo =='NV' or mode_wo == 'EV':
            vNew = vector_delta
            nVert = bm.verts.new(vNew)
            if mode_wo == 'EV':
                eVert = verts[-1]
                nEdge = bm.edges.new([eVert,nVert])
            bmesh.update_edit_mesh(obj.data)

class precision_draw_tools_cm(Operator):
    """Precision Drawing Tools"""
    bl_idname = "pdt.precision_draw_tools_cm"
    bl_label = "Precison Drawing Tools"
    bl_options = {'REGISTER', 'UNDO'}

    mode_wo: bpy.props.EnumProperty(
        items=(('CU', "Cursor", "Position Cursor"),
               ('MV', "Move", "Move Vertices/Objects"),
               ('NV', "Place Vert", "Place New Vertex"),
               ('EV', "Extrude Vert", "Extrude Active Vertex")),
        name="Operation Mode", default='CU')

    mode_pl: bpy.props.EnumProperty(
        items=(('XZ', "X-Z", "XZ Plane"),
               ('XY', "X-Y", "XY Plane"),
               ('YZ', "Y-Z", "YZ Plane")),
        name="Working Plane", default='XZ')

    mode_op1: bpy.props.EnumProperty(
        items=(('ABSOLUTE', "Global", "XY Coords"),
               ('DELTA', "Delta", "Delta Coords"),
               ('DISTANCE', "Dis/Ang", "Distance @ Angle Coords"),
               ('PERCENT', "Percent", "% -Uses Last 2 Select Vertices"),
               ('NORMAL', "Normal", "Normal -Uses Last 3 Selected Vertices"),
               ('INTERSECT', "Intersect", "Intersect -Uses Last 4 Selected Vertices")),
        name="Controls", default='ABSOLUTE')

    mode_op2: bpy.props.EnumProperty(
        items=(('NONE', "Controls", "Use Controls, not Functions"),
               ('ANGLE2', "Dis/Ang 2D", "Set Distance & Angle 2D -Uses Last 2 Select Vertices"),
               ('ANGLE3', "Dis/Ang 3D", "Set Distance & Angle 3D -Uses last 3 Selected Vertices"),
               ('JV', "Join 2 Verts", "Join 2 Vertices")),
        name="Functions", default='NONE')

    x_loc: bpy.props.FloatProperty(
        name="X", default=0.0, precision=5, step=1,
        description='X Coord')
    y_loc: bpy.props.FloatProperty(
        name="Y", default=0.0, precision=5, step=1,
        description='Y Coord')
    z_loc: bpy.props.FloatProperty(
        name="Z", default=0.0, precision=5, step=1,
        description='Z Coord')
    dis_v: bpy.props.FloatProperty(
        name="Dis", default=0.0, precision=5, step=1,
        description='Distance')
    ang_v: bpy.props.FloatProperty(
        name="Ang", default=0.0, precision=5, step=1,
        description='Angle')
    per_v: bpy.props.FloatProperty(
        name="%", default=0.0, precision=5, step=1, min=0, max=500,
        description='Percent')
    flip_p: bpy.props.BoolProperty(
        name="Flip % Start", default=False,
        description="Flip % To Non Active Vertex")
    flip_a: bpy.props.BoolProperty(
        name="Flip Angle", default=False,
        description="Flip Angle 180 degrees")
    trim_b: bpy.props.BoolProperty(
        name="Trim/Extend Both", default=False,
        description="Trim/Extend Vertices to Intersection")
    comm_v: bpy.props.StringProperty(
        name = "Command", default = "",
        description="Enter Processing Command")

    def draw(self,context):
        layout = self.layout
        row = layout.row()
        row.label(text='Operating Mode')
        row = layout.row()
        row.prop(self, 'mode_wo', expand=True)
        row = layout.row()
        row.label(text='Working Plane')
        row = layout.row()
        row.prop(self, 'mode_pl', expand=True)
        row = layout.row()
        row.label(text='Functions')
        row = layout.row()
        row.prop(self, 'mode_op2', expand=True)
        row = layout.row()
        row.label(text='Controls')
        row = layout.row()
        row.prop(self, 'mode_op1', expand=True)
        row = layout.row()
        row.label(text='Variables')
        row = layout.row()
        col = row.column()
        col.prop(self,'x_loc')
        col = row.column()
        col.prop(self,'dis_v')
        row = layout.row()
        col = row.column()
        col.prop(self,'y_loc')
        col = row.column()
        col.prop(self,'ang_v')
        row = layout.row()
        col = row.column()
        col.prop(self,'z_loc')
        col = row.column()
        col.prop(self,'per_v')
        row = layout.row()
        col = row.column()
        col.prop(self,'flip_p')
        col = row.column()
        col.prop(self,'flip_a')
        col = row.column()
        col.prop(self,'trim_b')

    def setMode(self, mode_pl):
        if mode_pl == 'XY':
            # a1 = x a2 = y a3 = z
            return 0, 1, 2
        elif mode_pl == 'XZ':
            # a1 = x a2 = z a3 = y
            return 0, 2, 1
        elif mode_pl == 'YZ':
            # a1 = y a2 = z a3 = x
            return 1, 2, 0

    def checkSelection(self, num, bm, obj):
        actE = bm.select_history[-1]
        if isinstance(actE, bmesh.types.BMVert):
            actV = actE.co
            if num == 1:
                return actV
            elif num == 2:
                othV = bm.select_history[-2].co
                return actV, othV
            elif num == 3:
                othV = bm.select_history[-2].co
                lstV = bm.select_history[-3].co
                return actV, othV, lstV
            elif num == 4:
                othV = bm.select_history[-2].co
                lstV = bm.select_history[-3].co
                fstV = bm.select_history[-4].co
                return actV, othV, lstV, fstV
        else:
            for f in bm.faces:
                f.select_set(False)
            for e in bm.edges:
                e.select_set(False)
            for v in bm.verts:
                v.select_set(False)
            bmesh.update_edit_mesh(obj.data)
            return None

    def execute(self, context):
        move_cursor(self, context, self.mode_wo, self.mode_op1, self.mode_op2, self.mode_pl, self.x_loc, self.y_loc,
            self.z_loc, self.dis_v, self.ang_v,self.per_v, self.flip_p, self.flip_a, self.trim_b, self.comm_v)
        return {'FINISHED'}

class precision_draw_tools_bm(Operator):
    """Precision Drawing Tools"""
    bl_idname = "pdt.precision_draw_tools_bm"
    bl_label = "Precison Drawing Tools - Command"
    bl_options = {'REGISTER', 'UNDO'}

    mode_wo: bpy.props.EnumProperty(
        items=(('CU', "Cursor", "Position Cursor"),
               ('MV', "Move", "Move Vertices/Objects"),
               ('NV', "Place Vert", "Place New Vertex"),
               ('EV', "Extrude Vert", "Extrude Active Vertex")),
        name="Operation Mode", default='CU')

    mode_pl: bpy.props.EnumProperty(
        items=(('XZ', "X-Z", "XZ Plane"),
               ('XY', "X-Y", "XY Plane"),
               ('YZ', "Y-Z", "YZ Plane")),
        name="Working Plane", default='XZ')

    mode_op1: bpy.props.EnumProperty(
        items=(('ABSOLUTE', "Global", "XY Coords"),
               ('DELTA', "Delta", "Delta Coords"),
               ('DISTANCE', "Dis/Ang", "Distance @ Angle Coords"),
               ('PERCENT', "Percent", "% -Uses Last 2 Select Vertices"),
               ('NORMAL', "Normal", "Normal -Uses Last 3 Selected Vertices"),
               ('INTERSECT', "Intersect", "Intersect -Uses Last 4 Selected Vertices")),
        name="Controls", default='ABSOLUTE')

    mode_op2: bpy.props.EnumProperty(
        items=(('RESET', "Reset", "Clear Variables"),
               ('ANGLE2', "Dis/Ang 2D", "Set Distance & Angle 2D -Uses Last 2 Select Vertices"),
               ('ANGLE3', "Dis/Ang 3D", "Set Distance & Angle 3D -Uses last 3 Selected Vertices"),
               ('JV', "Join 2 Verts", "Join 2 Vertices")),
        name="Functions", default='RESET')

    x_loc: bpy.props.FloatProperty(
        name="X", default=0.0, precision=5, step=1,
        description='X Coord')
    y_loc: bpy.props.FloatProperty(
        name="Y", default=0.0, precision=5, step=1,
        description='Y Coord')
    z_loc: bpy.props.FloatProperty(
        name="Z", default=0.0, precision=5, step=1,
        description='Z Coord')
    dis_v: bpy.props.FloatProperty(
        name="Dis", default=0.0, precision=5, step=1,
        description='Distance')
    ang_v: bpy.props.FloatProperty(
        name="Ang", default=0.0, precision=5, step=1,
        description='Angle')
    per_v: bpy.props.FloatProperty(
        name="%", default=0.0, precision=5, step=1, min=0, max=500,
        description='Percent')
    flip_p: bpy.props.BoolProperty(
        name="Flip % Start", default=False,
        description="Flip % To Non Active Vertex")
    flip_a: bpy.props.BoolProperty(
        name="Flip Angle", default=False,
        description="Flip Angle 180 degrees")
    trim_b: bpy.props.BoolProperty(
        name="Trim/Extend Both", default=False,
        description="Trim/Extend Vertices to Intersection")
    comm_v: bpy.props.StringProperty(
        name = "Command", default = "",
        description="Enter Processing Command")
    stor_v: bpy.props.StringProperty(
        name = "", default = "",
        description="Stored Values")

    def draw(self,context):
        layout = self.layout
        row = layout.row()
        row.label(text='Operating Mode')
        row = layout.row()
        row.prop(self, 'mode_wo', expand=True)
        row = layout.row()
        row.label(text='Working Plane')
        row = layout.row()
        row.prop(self, 'mode_pl', expand=True)
        row = layout.row()
        row.label(text='Functions')
        row = layout.row()
        row.prop(self, 'mode_op2', expand=True)
        row = layout.row()
        col = row.column()
        col.prop(self,'flip_p')
        col = row.column()
        col.prop(self,'flip_a')
        col = row.column()
        col.prop(self,'trim_b')
        #row = layout.row()
        #row.prop(self,'stor_v')
        row = layout.row()
        row.prop(self,'comm_v')

    def setMode(self, mode_pl):
        if mode_pl == 'XY':
            # a1 = x a2 = y a3 = z
            return 0, 1, 2
        elif mode_pl == 'XZ':
            # a1 = x a2 = z a3 = y
            return 0, 2, 1
        elif mode_pl == 'YZ':
            # a1 = y a2 = z a3 = x
            return 1, 2, 0

    def checkSelection(self, num, bm, obj):
        actE = bm.select_history[-1]
        if isinstance(actE, bmesh.types.BMVert):
            actV = actE.co
            if num == 1:
                return actV
            elif num == 2:
                othV = bm.select_history[-2].co
                return actV, othV
            elif num == 3:
                othV = bm.select_history[-2].co
                lstV = bm.select_history[-3].co
                return actV, othV, lstV
            elif num == 4:
                othV = bm.select_history[-2].co
                lstV = bm.select_history[-3].co
                fstV = bm.select_history[-4].co
                return actV, othV, lstV, fstV
        else:
            for f in bm.faces:
                f.select_set(False)
            for e in bm.edges:
                e.select_set(False)
            for v in bm.verts:
                v.select_set(False)
            bmesh.update_edit_mesh(obj.data)
            return None

    def execute(self, context):
        move_cursor(self, context, self.mode_wo, self.mode_op1, self.mode_op2, self.mode_pl, self.x_loc, self.y_loc,
            self.z_loc, self.dis_v, self.ang_v,self.per_v, self.flip_p, self.flip_a, self.trim_b, self.comm_v)
        return {'FINISHED'}

# Registration
addon_keymaps = []

def register():
    bpy.utils.register_class(precision_draw_tools_bm)
    bpy.utils.register_class(precision_draw_tools_cm)
    #bpy.types.VIEW3D_MT_edit_mesh_vertices.append(draw_menu)
    # handle the keymap
    wm = bpy.context.window_manager
    # Note that in background mode (no GUI available), keyconfigs are not available either,
    # so we have to check this to avoid nasty errors in background case.
    kc = wm.keyconfigs.addon
    if kc:
        km = wm.keyconfigs.addon.keymaps.new(name='Window', space_type='EMPTY')
        kmi = km.keymap_items.new(precision_draw_tools_cm.bl_idname, type = 'P', value = 'PRESS', ctrl=False, shift=True)
        addon_keymaps.append((km, kmi))
        km = wm.keyconfigs.addon.keymaps.new(name='Window', space_type='EMPTY')
        kmi = km.keymap_items.new(precision_draw_tools_bm.bl_idname, type = 'P', value = 'PRESS', ctrl=True, shift=True)
        addon_keymaps.append((km, kmi))

def unregister():
    # handle the keymap
    for km, kmi in addon_keymaps:
        km.keymap_items.remove(kmi)
    addon_keymaps.clear()
    bpy.utils.unregister_class(precision_draw_tools_cm)
    bpy.utils.unregister_class(precision_draw_tools_bm)
    #bpy.types.VIEW3D_MT_edit_mesh_vertices.remove(draw_menu)

if __name__ == "__main__":
    register()
